#!/usr/bin/env python
"""
This is performed on a VC or PSC in a particular SSO site.  It only needs to be run once per SSO site.
To determine the SSO site for a particular VC or PSC, you can run the following command:

VCSA:
    /usr/lib/vmware-vmafd/bin/vmafd-cli get-site-name --server-name localhost
Windows:
    "%VMWARE_CIS_HOME%"\vmafdd\vmafd-cli get-site-name --server-name localhost

The script automatically detects the local SSO site. 

Assumptions:

The mismatched ssltrust values are in the same SSO site as the node from which you run this.
There are plethora of reasons this could happen.

This option performs the following tasks:

    -- Checks for logbrowser:logbrowser and prompts for permission to remove it
    
    -- Checks for 5.x web client service and prompts for permission to remove it
    
    -- For each service found in the local SSO site, it compares the certificate presented 
       on 443 for the hostname in the URL with the sslTrust value in the service.  If they don't match, 
       it changes the SSLtrust and re-registers the service.  Otherwise, it skips the service.
"""

from .utils import *
from .utils import _getSslCert
import logging

logger = logging.getLogger(__name__)

class trustChecks(object):
    """
    Checks and corrects any service registration endpoints that contain an ssl trust
    value that doesn't match the certificate presented on 443.  Also checks for stale 
    5.5 webclient, logbrowser, and vcenterserver service registrations.
    
    Attributes:
        ls (lsClient): LookupServiceClientHelper object.
        services (dict): dictionary of services.
    """
    def __init__(self, params, username, password):
        """        
        Args:
            params (dict): Dictionary of local node specific parameters
            username (str): Username to auth with LS
            password (str): Password for the user specified
        """
        self.ls = LookupServiceClientHelper(params['psc'], username=username, password=password)
        # Gets all service registrations in the SSO domain
        self.services = self.ls.getSite(params['siteid'])
        logger.info("Retrieved services from SSO site: %s" % params['siteid'])

    def yesno_prompt(self,svcid):
        """
        prompt for yes or no and execute changes.
        
        Args:
            svcid (str): service ID to unregister
        """
        options = ['y','n']
        msg = "Would you like to unregister %s?[y/n]" % svcid
        try:
            answer = raw_input(msg)
        except:
            answer = input(msg)
        
        if answer.lower() not in options:
            logger.info("The option you specified is not supported.  Try again.")
            self.yesno_prompt(msg,options)
        else:
            if answer == 'y':
                logger.info("Attempting to unregister %s..." % svcid)
                self.ls.unregister(svcid)
                logger.info("Done.")
            if answer == 'n':
                logger.info("skipping...")
    def get_nodeid_mapping(self):
        mapping = {}
        for service in self.services:
            if service['serviceType']['type'] == 'vcenterserver':
                nodeid = service['nodeId']
                for endpoint in service['serviceEndpoints']:
                    if endpoint.get('url'):
                        parsed_url = urlparse.urlparse(endpoint['url'])
                        if parsed_url.hostname == 'localhost':
                            continue
                        else:
                            mapping.update({nodeid: parsed_url.hostname})
        return mapping

    def replaceService(self, servicetype):
        rebuild = Rebuilder(self.params, self.username, self.password)
        template = rebuild.getTemplate()
        try:
            servicedata, servicetype  = rebuild.serviceSelect(template, servicetype)
        except:
            logger.error("Couldn't find the service type %s in the specified template!")
            raise
        try:
            rebuild.rebuild_single_service(servicedata, servicetype)
        except Exception as e:
            logger.error("Failed to rebuild service %s!  Error was: %s" % (servicetype, e))

    def findAndFix(self):
        """
        This function will sort through all the services (dict) and evaluate the following in order:
        -- Is the service type a stale 5.x registration?  If so, prompt to delete.
        -- Compare ssltrust value of the service with the SSL certificate returned by _getSslCert
            - If the trust differs, update the value and re-register it.
            - If we can't get the cert from the host on port 443, we will blacklist it and ignore.
        """
        logger.info("Checking services for trust mismatches...")
        userwarning = False
        badtrust_count = 0
        host_blacklist = []
        nodeid_mapping = self.get_nodeid_mapping()
        for service in self.services:
            
            hostname = ""
            ssltrust = ""
            badtrust_flag = False

            serviceType = service['serviceType']['type']
            serviceId = service['serviceId']
            serviceVersion = service['serviceVersion']
            serviceOwner = service['ownerId']
            serviceNodeId = service['nodeId']
            
            if serviceType == 'logbrowser:logbrowser':
                logger.warning("logbrowser service registration detected!  this should only exist in 5.x!")
                self.yesno_prompt(serviceId)
                continue

            if serviceType == 'vsphereclient':
                if serviceVersion == '5.5' or serviceVersion == '5.1':
                    logger.warning("Service type %s with ID %s has been detected as a 5.x service.  This shouldn't exist!" % (serviceType,serviceId))
                    self.yesno_prompt(serviceId)
                    continue
                
                if 'WebClient' in serviceOwner:
                    logger.info("5.X solution user detected in a 'modern' service registration.  If you delete it in error, you can recreate it with lsdoctor.py -r.")
                    self.yesno_prompt(serviceId)
                    continue
            
            if serviceType == 'vcenterserver':
                if serviceVersion == '5.5' or serviceVersion == '5.1':
                    logger.warning("Service type %s with ID %s has been detected as a 5.x service.  This shouldn't exist!" % (serviceType,serviceId))
                    self.yesno_prompt(serviceId)
                    continue
                
                if 'vCenterServer' in serviceOwner:
                    logger.info("5.X solution user detected in a 'modern' service registration.  If you delete it in error, you can recreate it with lsdoctor.py -r.")
                    self.yesno_prompt(serviceId)
                    continue
            
            if service['serviceEndpoints']:
                for endpoint in service['serviceEndpoints']:
                    
                    if 'sslTrust' in endpoint.keys() and len(endpoint['sslTrust']) > 0:
                        
                        parsed_url = urlparse.urlparse(endpoint['url'])
                        if parsed_url.hostname == 'localhost':
                            if serviceNodeId in nodeid_mapping:
                                hostname = nodeid_mapping[serviceNodeId]
                            else:
                                continue
                        else:
                            hostname = parsed_url.hostname

                        if hostname not in host_blacklist:
                            try:
                                ssltrust = _getSslCert(hostname,443)
                            except:
                                logger.warning("%s is now blacklisted." % hostname)
                                host_blacklist.append(hostname)
                                break
                            if endpoint.get('sslTrust')[0].replace("\r\n","") != ssltrust:

                                badtrust_flag = True
                                userwarning = True
                                break
                        else:
                            break
            
            if badtrust_flag:
                badtrust_count += 1
                if ssltrust != "":
                    for endpoint in service['serviceEndpoints']:
                        endpoint['sslTrust'] = ssltrust
                    try:
                        logger.info("Attempting to reregister %s for %s" % (serviceId,hostname))
                        self.ls.unregister(serviceId)
                        self.ls.register(serviceId, service)
                    except:
                        logger.error("Failed to re-register %s" % serviceId)
                        raise
            else:
                logger.debug("%s service trust for %s is okay" % (serviceType, hostname))

        if userwarning == True:
            logger.info("We found %s mismatch(s) and fixed them :)" % badtrust_count)
        else:
            logger.info("No mismatches were found")

    def check(self):
        """
        Executes the check
        """
        self.findAndFix()


