#!/usr/bin/env python
"""
This performed on each PSC or VC in an SSO domain.

Assumptions:

if a system has been upgraded from 5.5, we can expect one or more of the following:
1.  Service registrations referencing an owner containing "WebClient" (for the vsphereclient service type) and 
    the version of the service registration is 5.x
2.  Service type "vcenterserver" has owner ID containing 'vCenterServer"
3.  Service type "logbrowser:logbrowser" exists
4.  If sso:sts, sso:groupcheck, and sso:admin reference "7444" in the service endpoints,
    it's likely they have a vecs store called "STS_INTERNAL_SSL_CERT".  The certificate in this store needs to match
    the MACHINE_SSL_CERT, and the service registrations need to be recreated with port 443.  These usually share comorbidity

This option performs the following tasks:

    -- Checks for and removes any service registration with service type: 'logbrowser:logbrowser' [lookup service API]
    
    -- Checks for and removes any service registration with service type 'vsphereclient' or 'vcenterserver' 
       and version '5.1' or '5.5' [lookup service API]
    
    -- Checks for and fixes any service registration with service type 'vsphereclient' and 
       an owner ID containing 'WebClient'.  An owner ID containing 'WebClient' is a 5.x solution user [lookup service API]
    
    -- Checks for any service registrations with ':7444' in the URL.  If so, it will check for and fix 
       STS_INTERNAL_SSL_CERT in VECS, then remove and recreate legacy SSO service registrations [vecs-cli and lookup service installer]
"""
import os
from .utils import *
from .rebuild import *
try:
    from packaging import version
except ImportError:
    # Won't exist in python 2.7
    version = None
try:
    import httplib
except ImportError:
    import http.client as httplib
try:
    import urllib.parse as urlparse
    from urllib.request import Request, urlopen
    from urllib.error import URLError, HTTPError
except ImportError:
    import urlparse
    from urllib2 import Request, urlopen
    from urllib2 import URLError, HTTPError

if os.name != 'posix':
    REREG_DIR = os.environ['VMWARE_CFG_DIR'] + "\\rereg\\"
else:
    REREG_DIR = "/etc/vmware/rereg/"

curpath = os.getcwd()

import logging
logger = logging.getLogger(__name__)

class VecsCheck(object):

    """
    Checks, backs up, and replaces STS_INTERNAL_SSL_CERT.  Also backs up the MACHINE_SSL_CERT.
    
    Attributes:
        machine_cert_file (str): desired path to MACHINE_SSL_CERT certificate file
        machine_key_file (str): desired path to MACHINE_SSL_CERT key file
        stores (list): list of vecs stores
        sts_internal_ssl_cert_file (str): desired path to STS_INTERNAL_SSL_CERT certificate file
        sts_internal_ssl_key_file (str): desired path to STS_INTERNAL_SSL_CERT certificate file
    """

    def __init__(self, exportpath):
        """
        Args:
            exportpath (str): path to the export directory
        """
        init_vecs = VecsStore()
        self.stores = init_vecs.list()
        if not os.path.exists(exportpath):
            os.makedirs(exportpath)


        # set certificate filenames for use later.
        self.machine_cert_file = os.path.join(exportpath, "MACHINE_SSL_CERT.crt")
        self.machine_key_file = os.path.join(exportpath, "MACHINE_SSL_CERT.key")
        self.sts_internal_ssl_cert_file = os.path.join(exportpath, "STS_INTERNAL_SSL_CERT.crt")
        self.sts_internal_ssl_key_file = os.path.join(exportpath, "STS_INTERNAL_SSL_CERT.key")

    def export_cert_key(self, store, alias, cert_file, key_file):
        """Summary
        
        Args:
            store (str): name of the vecs store to go look in
            alias (str): alias of the cert go grab
            cert_file (str): path to the desired export location for the certificate
            key_file (str): path to the desired export location for the key
        
        Returns:
            str: Returns cert file and key file names
        """

        vecs_entry = VecsEntry(store)
        try:
            vecs_entry.get_cert(alias,cert_file)
        except:
            logger.error("couldn't write certificate %s/%s to file!" % (store,alias))
            raise
        try:
            vecs_entry.get_key(alias,key_file)

        except:
            logger.error("couldn't write key %s/%s to file!" % (store,alias))
            raise
        return cert_file, key_file

    def backup_machine(self):
        """
        Calls export_cert_key to make it more intuitive to use
        
        Returns:
            str: Returns cert file and key file names
        """
        # backup MACHINE_SSL_CERT
        logger.info("Exporting MACHINE_SSL_CERT cert and key")
        cert,key = self.export_cert_key('MACHINE_SSL_CERT', 
                                        "__MACHINE_CERT", 
                                        self.machine_cert_file, 
                                        self.machine_key_file)
        return cert,key

    def backup_sts_internal(self):
        """
        Calls export_cert_key to make it more intuitive to use
        
        Returns:
            str: Returns cert file and key file names
        """
        # backup STS_INTERNAL_SSL_CERT
        logger.info("Backing up STS_INTERNAL_SSL_CERT")
        cert,key = self.export_cert_key('STS_INTERNAL_SSL_CERT',
                                        '__MACHINE_CERT',
                                        self.sts_internal_ssl_cert_file,
                                        self.sts_internal_ssl_key_file)
        return cert,key
    
    def replace_sts_internal(self):
        """
        Delete the certificate in STS_INTERNAL_SSL_CERT store and replace with MACHINE_SSL_CERT 
        previously exported.
        """

        logger.info("Replacing STS_INTERNAL_SSL_CERT with MACHINE_SSL_CERT")

        try:
            sts_internal = VecsEntry('STS_INTERNAL_SSL_CERT')
            sts_internal.delete('__MACHINE_CERT')
        except:
            logger.error("Couldn't delete the STS_INTERNAL_SSL_CERT!")
            raise
        try:
            sts_internal.create('__MACHINE_CERT', self.machine_cert_file, self.machine_key_file)
        except:
            logger.error("Couldn't create new STS_INTERNAL_SSL_CERT!")
            raise
        logger.info("Successfully replaced STS_INTERNAL_SSL_CERT")
    
    def check_sts_internal(self):
        """
        Check to see if STS_INTERNAL_SSL_CERT exists, and replace if it does.
        
        Returns:
            boolean: True if the store exists.
        """
       

        self.backup_machine()

        result = False
        
        logger.info("Checking for STS_INTERNAL_SSL_CERT...")
        
        if 'STS_INTERNAL_SSL_CERT' in self.stores:
            result = True
            self.backup_sts_internal()
        return result


class staleChecks(object):

    """
    Gets all service registrations for this specific node, and runs a check.
    
    Attributes:
        cert_file (str): path to the certificate file
        domain_name (str): SSO domain name
        exportpath (str): Path to export location
        ls (LookupServiceClientHelper object): LS client session
        machine_id (str): machine ID of the current node
        pnid (str): Primary network identifier (FQDN)
        ssltrust (str): SSL trust value current present on port 443
        noprompt (bool): whether to suppress all user prompting for automation
        template_file_override (str): path to specified template file
    """
    def __init__(self, params, username, password,
                 exportpath=curpath, noprompt=False, template_file=None):
        """
        Args:
            params (dict): local node parameters
            username (str): Username with which to auth to LS
            password (str): password for user specified
            exportpath (str, optional): path to export directory
            noprompt (bool): whether to suppress all user prompting for
                             automation purposes
            template_file (str, optional): path to template file to use if a
                                           service needs to be rebuilt
        """
        self.params = params
        self.exportpath = exportpath
        self.machine_id = params['machineid']
        self.domain_name = params['domain_name']
        self.ssltrust = params['ssltrust']
        self.pnid = params['pnid']
        self.username = username
        self.password = password
        self.noprompt = noprompt
        self.template_file_override = template_file

        self.ls = LookupServiceClientHelper(params['psc'], username=username,
                                            password=password)
        
        # getServices only returns services for this node.
        self.services, hostid = self.ls.getPnid(self.pnid)
        logger.info("Retrieved services for machine with hostname: %s" % self.pnid)
    
    def replaceOwnerID(self,service):
        """
        Replaces the Owner ID parameter in a stale service.  Also replaces the service 
        ID parameter in a stale service obtained from rereg directory.
        
        Args:
            service (dict): service registration as a dictionary
        """
        serviceId = service['serviceId']
        service['ownerId'] = 'vsphere-webclient-%s@%s' % (self.machine_id, self.domain_name)
        serviceType = service['serviceType']['type']
        newServiceId = self.getServiceId(serviceType)
        self.ls.unregister(serviceId)
        service['serviceId'] = newServiceId
        self.ls.register(newServiceId,service)
    def getServiceId(self, servicetype):
        """
        Get the original service ID of the given service type for the current node.
        This is obtained by looking into the services located in /etc/vmware/rereg.

        Args:
            servicetype (str): Service type
        
        Returns:
            str: Service ID
        """
        serviceid = ""
        servicetypename = "serviceType.type=" + servicetype
        servicemap = {}

        # search files in rereg directory.  
        files = [fn for fn in os.listdir(REREG_DIR) if fn.endswith('.prop')]
        for filename in files:
            with open(REREG_DIR + filename) as fn:
                data = fn.read()
                data = data.replace(" ","")

                for line in data.splitlines():
                    
                    # Search for provided service type, return
                    # the name of the file minus _spec.prop as this
                    # will be the original service ID for that service.
                    if line == servicetypename:
                        serviceid = filename.replace("_spec.prop","")
                        break

        
        # return the service ID                                 
        return serviceid

    def replaceService(self, servicetype):
        if self.noprompt:
            logger.debug("Running 'replaceService' for servicetype '%s' as an "
                         "automated rebuild service action without user "
                         "prompts" % servicetype)
            rebuild = Rebuilder(self.params, self.username, self.password,
                                rebuild_action=ACTION_REBUILD_SERVICE,
                                service_name=servicetype,
                                template_file=self.template_file_override)
            rebuild.run_action()
            return

        rebuild = Rebuilder(self.params, self.username, self.password)
        template = rebuild.getTemplate()
        try:
            servicedata, servicetype  = rebuild.serviceSelect(template, servicetype)
        except:
            logger.error("Couldn't find the service type %s in the specified template!  Going to move on for now..." % servicetype)
        try:
            rebuild.rebuild_single_service(servicedata, servicetype)
        except Exception as e:
            logger.error("Failed to rebuild service %s!" % servicetype)
            logger.debug("Error was: %s" % e)


    def checkLegacy(self):
        """
        This checks for port 7444 in the 'legacy sso endpoints'.
        The endpoints are sso:sts, sso:groupcheck, and sso:admin.  port 7444 and 
        the presence of STS_INTERNAL_SSL_CERT share a comorbidity, so we will try to update
        STS_INTERNAL_SSL_CERT with the MACHINE_SSL_CERT and then regenerate the legacy SSO endpoints.

        """
        
        legacy_sso_services = []
        legacyflag = False

        vecs_check = VecsCheck(self.exportpath)
        vecs_check.backup_machine()
        self.cert_file = vecs_check.machine_cert_file
        logger.info("Checking for STS_INTERNAL_SSL_CERT...")
        if vecs_check.check_sts_internal():
            logger.info("PROBLEM FOUND:  STS_INTERNAL_SSL_CERT found!")
            vecs_check.replace_sts_internal()
        else:
            logger.info("No STS_INTERNAL_SSL_CERT store found.")

        logger.info("Checking for 7444 in legacy services...")
        # loop through services and check for any reference to port 7444, record the service ID
        for x in self.services:
            serviceType = x['serviceType']['type']
            serviceId = x['serviceId']
            serviceVersion = x['serviceVersion']
            for endpoint in x['serviceEndpoints']:
                url = urlparse.urlparse(endpoint.get('url'))
                if url.port == 7444:
                    logger.warning("PROBLEM FOUND: Found port 7444 in service registration URL!  %s" % endpoint.get('url'))
                    legacyflag = True
                    legacy_sso_services.append(serviceId)
        
        # if we found anything with 7444, unregister them and recreate the legacy SSO services
        if legacyflag:
            for service in legacy_sso_services:
                self.ls.unregister(service)

            logger.info("Recreating legacy SSO service registrations...")
            regen = recreateSSO(self.pnid,self.cert_file)
            if regen.legacy() == 1:
                regen = recreateSSO(self.pnid,self.cert_file)
                if regen.legacy() == 0:
                    logger.info("Successfully recreated legacy SSO endpoints.")
            else:
                logger.info("Successfully recreated legacy SSO endpoints.")
                
        else:
            logger.info("PASSED: Port 7444 not found in registrations.")
    
    def checkStale(self):
        """
        here, we loop through all the services and look for the following:
            -- if the service type is "logbrowser:logbrowser", unregister it.
            -- if the service type is "vsphereclient" but has service version 5.x, unregister it.
                - if the vsphereclient service has a 6.x version, but OwnerID containing "WebClient",
                  replace the owner ID with the correct vsphere-webclient solution user value, then
                  replace the service ID with the original from rereg directory.
            -- if the service type is "vcenterserver" but has service version 5.x, unregister it.
        """
        logging.info("Checking for logbrowser or 5.x vsphere client services...")
        logbrowserids = []
        webclientids = []
        vcenterids = []
        clientreplaceflag = False
        vcenterreplaceflag = False
        for x in self.services:
            serviceType = x['serviceType']['type']
            serviceId = x['serviceId']
            serviceVersion = x['serviceVersion']
            serviceOwner = x['ownerId']
            
            # check for 'Service Type: logbrowser:logbrowser'
            if serviceType == "logbrowser:logbrowser":
                logbrowserids.append(serviceId)
            
            # find vsphereclient service registration that has 'Version: 5.x'
            if serviceType == 'vsphereclient':
                if serviceVersion == '5.5' or serviceVersion == '5.1':
                    webclientids.append(serviceId)
                    continue
                if 'WebClient' in serviceOwner:
                    logger.info("PROBLEM FOUND: 5.X solution user detected in a 'modern' service registration.  Attempting to correct and re-register.")
                    webclientids.append(serviceId)

            if serviceType == 'vcenterserver':
                if serviceVersion == '5.5' or serviceVersion == '5.1':
                    vcenterids.append(serviceId)
                    continue
                if 'vCenterServer' in serviceOwner:
                    logger.info("PROBLEM FOUND: 5.X solution user detected in a 'modern' service registration.  Attempting to correct and re-register.")
                    vcenterreplaceflag = True

        if len(logbrowserids) > 0:
            logger.warning("PROBLEM FOUND: logbrowser service found.  Attempting to unregister...")
            for entry in logbrowserids:
                self.ls.unregister(entry)
            logger.info("Success!")
        else:
            logger.info("PASSED: logbrowser service not found.")
        
        if len(webclientids) > 0:
            if  version is None or version.parse(self.params['version']) < version.parse("7.0") :
                self.replaceService("vsphereclient")
            else:
                logger.warning("PROBLEM FOUND: stale 5.x webclient service found.  Attempting to unregister...")
                for entry in webclientids:
                    self.ls.unregister(entry)
                logger.info("Success!")
        else:
            logger.info("PASSED: 5.x webclient service not found.")
        
        if vcenterreplaceflag:
            self.replaceService("vcenterserver")
        else:
            if len(vcenterids) > 0:
                logger.warning("PROBLEM FOUND: stale 5.x vcenter service found.  Attempting to unregister...")
                for entry in vcenterids:
                    self.ls.unregister(entry)
                logger.info("Success!")
            else:
                logger.info("PASSED: 5.x vcenter service not found.")

    def check(self):
        """
        Executes the check.
        """
        self.checkStale()
        self.checkLegacy()
