#!/usr/bin/env python
"""
Used to recreate all the solution users for the node.  Assumes
that the solution user certificates still exist in VECS, and that
the solution users don't already exist in vmdir.  Will check the 
permissions of each solution user and correct them if necessary.
"""

from .utils import *
import logging

logger = logging.getLogger(__name__)

class solutionusersCheck(object):
    """
    Class for recreating solution users on a node

    """
    def __init__(self, username, password):
        """
        Args:
            username (str): SSO admin username
            password (str): SSO admin password
        """
        self.username = username
        self.password = password

    def checkAndFix(self):
        """
        This class does the heavy lifting.  It defines the standard user groups
        for the known solution users.  It will delete the solution users and it will also build the solution user data based on
        the params.  Then, it will perform the following in order:
        -- list all solution users
        -- delete and recreate all the users, doing the following:
            - use dir-cli to delete the user from vmdir
            - Get certificate from VECS
            - use dir-cli util to create the solution user in vmdir
            - add the user to the appropriate groups as predefined.
        """
        params = get_params()
        #INITIALIZE VARIABLES
        machine = params.get()['pnid']
        deploytype = params.get()['deploytype']
        machineid = params.get()['machineid']
        domain = params.get()['domain_name']
        user_groups = {}
        user_groups['vpxd'] = "ComponentManager.Administrators,SystemConfiguration.Administrators,LicenseService.Administrators"
        user_groups['vpxd-extension'] = "ComponentManager.Administrators,SystemConfiguration.Administrators"
        user_groups['machine'] = "ComponentManager.Administrators,SystemConfiguration.Administrators"
        user_groups['vsphere-webclient'] = "LicenseService.Administrators"
        user_groups['hvc'] = "ComponentManager.Administrators,LicenseService.Administrators,SystemConfiguration.Administrators"
        user_groups['wcp'] = "NsxAdministrators"


        global_groups = {}
        global_groups['vpxd'] = "Users,ComponentManager.Administrators,SystemConfiguration.Administrators,LicenseService.Administrators"
        global_groups['vpxd-extension'] = "Users,ActAsUsers,ComponentManager.Administrators,SystemConfiguration.Administrators"
        global_groups['machine'] = "Administrators,ComponentManager.Administrators,SystemConfiguration.Administrators"
        global_groups['vsphere-webclient'] = "Administrators,ActAsUsers,LicenseService.Administrators"
        
        if deploytype == "infrastructure":
            user_types= ["machine","vsphere-webclient"]
        else:
            user_types= ["vpxd","vpxd-extension","machine","vsphere-webclient","hvc","wcp"]
        dirclient = DirClient()
        dirclient_service_params = {}
        dirclient_service_params['machineid'] = machineid
        dirclient_service_params['username'] = self.username
        dirclient_service_params['password'] = self.password
        
        #GET SERVICES
        services = dirclient.list_services(self.password).split()
        localservices = [i for i in services if machineid in i]
        #CHECK IF USERS EXIST, IF THEY DON'T, RECREATE   
        flag = False
        for user_type in user_types:
            dirclient_service_params['usertype'] = user_type
            solutionuser=user_type + "-" + machineid
            logging.info("Deleting and recreating user " + user_type)
            dirclient.delete_service(dirclient_service_params)
            #GET CERTIFICATE
            vecs = VecsEntry(user_type)
            certname = user_type + ".crt"
            dirclient_service_params['certpath'] = os.path.join(os.getcwd(),certname)
            vecs.get_cert(user_type, dirclient_service_params['certpath'])
            #COMPLETE dirclient_service_params object
            dirclient_service_params['ssogroups'] = user_groups[user_type]
            if user_type == "vsphere-webclient":
                dirclient_service_params['ssoadminrole'] = "Administrator"
                dirclient_service_params['wstrustrole'] = "ActAsUser"
                dirclient.create_service(dirclient_service_params)
                del dirclient_service_params['wstrustrole']
            
            if user_type == "vpxd-extension":
                dirclient_service_params['ssoadminrole'] = "User"
                dirclient_service_params['wstrustrole'] = "ActAsUser"
                dirclient.create_service(dirclient_service_params)
                del dirclient_service_params['wstrustrole']
                
            if user_type == "machine":
                dirclient_service_params['ssoadminrole'] = "Administrator"
                dirclient.create_service(dirclient_service_params)
                 
            if user_type == "vpxd":
                dirclient_service_params['ssoadminrole'] = "User"
                dirclient.create_service(dirclient_service_params)

            if user_type == "wcp":
                dirclient_service_params['ssoadminrole'] = "User"
                dirclient_service_params['wstrustrole'] = "ActAsUser"
                dirclient.create_service(dirclient_service_params)
                del dirclient_service_params['wstrustrole']

            if user_type == "hvc":
                dirclient_service_params['ssoadminrole'] = "User"
                dirclient.create_service(dirclient_service_params)
            
        logging.info("Execution finished - Please check log for details")
    