#!/usr/bin/env python
"""
Used to generate a template, build new services based on the candidate, 
replace all services, replace a single service, or restore to backup

Assumptions:

You have either a healthy node from which to pull a template, or
a template from the same build has already been provided in the templates 
directory.  Also assumes that all 2nd and 3rd party solutions that 
register with the lookup service will be re-registered at the conclusion
of the activity.

Menu:
        Version Detected
            Deployment type: embedded
            Version: 6.7.0.31000
        ========================

        0.  Exit
        1.  Generate a template.
        2.  Replace all services with new services.
        3.  Replace individual service.
        4.  Restore services from backup file.

        ========================

In most cases, the primary use will be option 2 and 3.

Option 1:

This option will generate a template for the current version.
Template location is the "templates" directory

Option 2:

This will replace all LS registrations with fresh ones based
on an available template.  New services are also output as a 
candidate file.  Candidate location is the "candidates" 
directory.

Option 3:

This will allow you to recreate any single non-SSO service
registration based on an available template.  Recreating the
SSO services requires use of option 2.  New services are also
output as a candidate file.  Candidate location is the 
"candidates" directory.

Option 4:

This will put the original non-SSO service registrations back.
SSO service registrations are freshly regenerated.  Backup 
location is the "backups" directory.
"""
import os
import uuid
from .utils import *
try:
    import httplib
except ImportError:
    import http.client as httplib
try:
    import urllib.parse as urlparse
    from urllib.request import Request, urlopen
    from urllib.error import URLError, HTTPError
except ImportError:
    import urlparse
    from urllib2 import Request, urlopen
    from urllib2 import URLError, HTTPError
from cis.defaults import get_cis_log_dir
from .utils import get_params

import logging
logger = logging.getLogger(__name__)

if os.name != 'posix':
    REREG_DIR = os.environ['VMWARE_CFG_DIR'] + "\\rereg\\"
else:
    REREG_DIR = "/etc/vmware/rereg/"

curpath = os.getcwd()
PSC_SERVICES = ["cs.identity","sso:sts","sso:admin","sso:groupcheck"]

ACTION_REBUILD_SERVICE = 'service'
ACTION_REBUILD_ALL_SERVICES = 'all_services'
ACTION_CREATE_TEMPLATE = 'create_template'
ACTION_RESTORE = 'restore'


def getVersion():
    """
    Attempts to get version from the /etc/issue file
    
    Args:
        return_full (bool, optional): return entire string from file if true

    Returns:
        str: version
    """
    params = get_params()
    params = params.get()

    full_version = "_".join([params['build'],
                            params['version'],
                            params['os_type']])

    return full_version

def _createDirs(dir_name):
    """
    Creates the specified directory
    
    Args:
        dir_name (str): name of the directory to create
    """
    if not os.path.exists(dir_name):
        os.makedirs(dir_name)

def _setDateFilename(dir_name, name, file_type):
    """
    Sets filename in a helpful format
    
    Args:
        dir_name (str): Directory
        name (str): file name
        file_type (str): file extension
    
    Returns:
        str: full path to time stamped file
    """
    file_name = str(time.strftime(name + "-%Y-%m-%d-%H%M%S" + file_type))
    return os.path.abspath(os.path.join(dir_name, file_name))

def _setOtherFilename(dir_name, name, file_type, expand=True):
    """
    Set filename
    
    Args:
        dir_name (str): Directory
        name (str): file name
        file_type (str): file extension
    
    Returns:
        str: full path to time stamped file
    """
    shortname = name + file_type
    file_name = os.path.abspath(os.path.join(dir_name, shortname))
    if expand:
        if os.path.isfile(file_name):
            expand = 0
            while True:
                expand += 1
                new_file_name = file_name.split(file_type)[0] + "_" + str(expand) + file_type
                if os.path.isfile(new_file_name):
                    continue
                else:
                    file_name = new_file_name
                    break
    return file_name

def loadJson(file):
    """
    Loads a JSON file and returns in healthy format
    
    Args:
        file (str): path to file
    
    Returns:
        str: JSON data
    """
    file = os.path.abspath(file)
    with open(file) as f:
        jsondata = json.load(f)
    return jsondata

def fileSelect(dir_name):
    """
    Provides menu to select a file

    Args:
        dir_name (str): directory of files from which to build the menu
    
    Returns:
        str: Path to selected file.
    """
    logger.info("Getting files from %s" % dir_name)
    print("Please select a file:\n")
    
    folder = dir_name
    file_names = [fn for fn in os.listdir(dir_name) if os.path.isfile(os.path.join(dir_name,fn))]
    file_names = [fn for fn in file_names if ".json" in fn]

    file_names.sort(key=lambda x: os.path.getmtime(os.path.abspath(os.path.join(dir_name,x))))

    count = -1

    for f in file_names:
        count = count + 1
        print("[%s] %s " % (str(count), f))

    while True:
        try:
            ans_file = input("Select number: ")
        except:
            ans_file = raw_input("Select number: ")

        if int(ans_file) > count:
            print("There is no entry for %s." % ans_file)
            return None

        path = os.path.abspath(os.path.join(dir_name, file_names[int(ans_file)]))
        logger.info("Selected file: %s " % path)
        break
    return path


class Rebuilder(object):
    """
    Generates and builds a full set of LS services based on templates.
    
    Attributes:
        backup_file (str): path to backup file
        backuppath (str): path to backup directory
        candidate_file (str): path to candidate file
        candidatepath (str): path to candidate directory
        cert_backup (str): path to the backup of MACHINE_SSL_CERT
        ls (LookupServiceClientHelper object): LS client object
        params (dict): dictionary of local node parameters
        services (dict): dictionary of services
        template (dict): dictionary containing template version of services
        template_file (str): path to template file
        templatepath (str): path to template directory
        version (str): version detected
        rebuild_action (str): rebuild action corresponding to menu choices
        service_override (str): specified service name to rebuild
        template_file_override (str): path to specified template file
        backup_file_override (str): specified path to backup file
    """
    def getServiceIdsFromSpecs(self):

        ignorelist = ['rereg', 'service-state' ,'example']
        results = {}
        for dirname, subdir, filelist in os.walk('/'):
            if 'rereg' not in dirname and 'service-state' not in dirname:
                if not any(item in ignorelist for item in filelist):
                    for file in filelist:
                        if 'prop' in file or 'spec' in file:
                            # print(file)
                            try:
                                filename = str(os.path.join(dirname,file))
                                filedata = open(filename).read().replace(' ','')

                                if 'cmreg.serviceid' in filedata:
                                    serviceType = ""
                                    serviceId = ""
                                    for line in filedata.splitlines():
                                        if 'serviceType.type=' in line:
                                            serviceType = line.split('=')[1]
                                    for line in filedata.splitlines():
                                        if 'cmreg.serviceid=' in line:
                                            serviceId = line.split('=')[1]
                                    if 'service-id' not in serviceId and 'service-uuid' not in serviceId:
                                        if serviceId != "" and serviceType != "":
                                            results[serviceType] = serviceId
                            except UnicodeDecodeError:
                                # print("BAD FILE %s" % file)
                                continue
                            except FileNotFoundError:
                                continue
        return results
    
    def __init__(self, params, username, password,
                 workingdir=curpath, rebuild_action=None, service_name=None,
                 template_file=None, restore_file=None):
        """
        Args:
            params (dict): dictionary of local node parameters
            username (str): SSO admin username
            password (str): SSO admin password
            workingdir (str, optional): optionally specify the working
                                        directory, otherwise is current path
            rebuild_action (str, optional): rebuild action to perform
            service_name (str, optional): specific service to rebuild
            template_file (str, optional): path to template file for rebuilding
            restore_file (str, optional): path to backup for for restoring
        """
        self.serviceIds = self.getServiceIdsFromSpecs()
        self.version = getVersion()
        self.backuppath = os.path.join(workingdir,'backups')
        self.candidatepath = os.path.join(workingdir,'candidates')
        self.templatepath = os.path.join(workingdir,'templates')
        self.cert_backup = os.path.join(self.backuppath,"machine.crt")
        self.params = params

        _createDirs(self.backuppath)
        _createDirs(self.candidatepath)
        _createDirs(self.templatepath)

        self.template_file = _setOtherFilename(self.templatepath,
                                               self.version, ".json")
        self.backup_file = _setDateFilename(self.backuppath, "backup", ".json")
        self.candidate_file = _setDateFilename(self.candidatepath,
                                               "candidate", ".json")

        self.rebuild_action = rebuild_action
        self.service_override = service_name
        self.template_file_override = template_file
        self.backup_file_override = restore_file

        psc_embedded = ['infrastructure', 'embedded']

        # establish LS client connection.  If first attempt fails, try to create
        # cs.identity service and try again.
        try:
            self.ls = LookupServiceClientHelper(self.params['psc'],
                                                username=username,
                                                password=password)
            logger.info("Established LS connection to %s" % self.params['psc'])
        except (sso.SoapException, Exception) as e:
            if 'Invalid credentials' in str(e):
                sys.exit()

            logger.error("Initial connection to lookup service failed.  "
                         "Will attempt to regenerate cs.identity and try again")
            try:
                cert = VecsEntry("MACHINE_SSL_CERT")
                exported = cert.get_cert("__MACHINE_CERT", self.cert_backup)
                
                if self.params['deploytype'] in psc_embedded:           
                    ssoentries = recreateSSO(self.params['pnid'],
                                             self.cert_backup)
                    logger.warning("Recreating SSO service registrations...")
                    if ssoentries.modern() == 1:
                        ssoentries = recreateSSO(self.params['pnid'],
                                                 self.cert_backup)
                        if ssoentries.modern() == 0:
                            logger.info("Successfully recreated modern "
                                        "SSO endpoints.")
                        else:
                            logger.exception("Failed to recreate modern "
                                             "SSO endpoints!")
                            raise
                    self.ls = LookupServiceClientHelper(self.params['psc'],
                                                        username=username,
                                                        password=password)
                    logger.info("Established LS connection to %s"
                                % self.params['psc'])
            except Exception as e:
                logger.exception("Still failed to connect to lookup service.  "
                                 "Further investigation required.  error: %s"
                                 % e)
                raise

    def getTemplate(self, prompt=True):
        """
        Returns a matching template.  If no matching template is found,
        prompts with menu to select a template file from the templates
        directory and returns the chosen one.  If the prompt flag is
        not asserted and the template file is not found, the user will
        not be prompted further and a runtime error will be raised

        Args:
            prompt (bool, optional): whether to prompt the user for a selection

        Returns:
            json: Built template in JSON format
        """
        self.template_file = _setOtherFilename(self.templatepath, self.version,
                                               ".json", expand=False)
        if self.template_file_override:
            candidate_file = self.template_file_override
        else:
            candidate_file = self.template_file

        logger.debug("Detected version %s.  Attempting to find template: %s "
                     % (self.version, candidate_file))
        if os.path.isfile(candidate_file):
            logger.info("Found %s" % candidate_file)
            return self.build_from_template(candidate_file)
        elif not prompt:
            logger.error("Could not find template file %s" % candidate_file)
            raise RuntimeError("Failed to load template %s" % candidate_file)
        else:
            print("\nNo template found for %s.  Proceeding to file select.\n"
                  % self.version)
            candidate_file = fileSelect(self.templatepath)
            if candidate_file:
                # newdata = candidate_file
                return self.build_from_template(candidate_file)
            else:
                logger.warning("invalid selection!  Try again.")
                self.menu()


    def unregisterServiceType(self, servicetype):
        """
        Unregisters all services of the specified service type
        
        Args:
            servicetype (str): specified service type to unregister
        """
        temp_svcs, hostid = self.ls.getPnid(self.params['pnid'])
        for service in temp_svcs:
            if servicetype in service['serviceType']['type']:
                try:
                    self.ls.unregister(service['serviceId'])
                    msg = "Unregistered %s" % service['serviceId']
                    logger.info(msg)
                except:
                    logger.error("FAILED TO UNREGISTER SERVICE!")
                    break
    
    def getServiceId(self, servicetype):
        """
        Get the original service ID of the given service type for the current node.
        This is obtained by looking into the services located in /etc/vmware/rereg.

        Args:
            servicetype (str): Service type
        
        Returns:
            str: Service ID
        """
        serviceid = ""
        servicetypename = "serviceType.type=" + servicetype
        servicemap = {}
        if servicetype in self.serviceIds.keys():
            serviceid = self.serviceIds.get(servicetype)
        else:
            # search files in rereg directory.  
            files = [fn for fn in os.listdir(REREG_DIR) if fn.endswith('.prop')]
            for filename in files:
                with open(REREG_DIR + filename) as fn:
                    data = fn.read()
                    data = data.replace(" ","")

                    for line in data.splitlines():
                        
                        # Search for provided service type, return
                        # the name of the file minus _spec.prop as this
                        # will be the original service ID for that service.
                        if line == servicetypename:
                            serviceid = filename.replace("_spec.prop","")
                            break

        # return the service ID                                 
        return serviceid

    def backup(self):
        """
        Backs up the MACHINE_SSL_CERT certificate, as well as the current version of 
        the nodes LS registrations.
        """
        # backup machine certificate
        cert = VecsEntry("MACHINE_SSL_CERT")
        exported = cert.get_cert("__MACHINE_CERT",self.cert_backup)
        
        # grab a backup of my services
        self.services, myId = self.ls.getPnid(self.params['pnid'])
        logger.debug("Retrieved services for machine with hostname: %s" % self.params['pnid'])

                # dump the backup to the file    
        with open(self.backup_file,"w") as f:
            json.dump(self.services,f,indent=1)
            logger.debug("Dumped backup of existing services to %s" % self.backup_file)

    def generate_template(self):
        """
        This allows us to build a template from the current node's services.
        It will replace it's unique values with parameter place-holders for use
        as a template.
        
        Returns:
            json: Template of LS services for this node
        """
        # get a backup and build the template.
        self.backup()
        
        # using the backup, find my 'vcenterserver' service ID for use later.
        for service in self.services:
            if "vcenterserver" in service['serviceType']['type']:
                my_vcserviceid = service.get('serviceId')
                self.params['vcserviceid'] = my_vcserviceid
        
        # PSC specific services must be regenerated, as the service IDs aren't recorded in rereg
        self.services = [x for x in self.services if x["serviceType"].get('type') not in PSC_SERVICES]
        
        # Map the service ID variable
        # some service IDs we can't get from rereg.  Luckily the service IDs are built
        # based on the 'vcenterserver' service ID with the service type appended.
        for service in self.services:
            if 'vcserviceid' in self.params:
                if self.params['vcserviceid'] in service['serviceId']:
                    service['serviceId'] = service.get('serviceId').replace(self.params['vcserviceid'], "{vcserviceid}")
                else:
                    service['serviceId'] = "{serviceid}"
            else:
                service['serviceId'] = "{serviceid}"
            
            # map the solution user variable
            service['ownerId'] = service['ownerId'].replace(self.params['machineid'], "{machineid}")
            service['ownerId'] = service['ownerId'].replace(self.params['domain_name'], "{domain_name}")

            # Sometimes ownerID is just the machine account
            if self.params['pnid'] in service['ownerId']:
                service['ownerId'] = service['ownerId'].replace(
                    self.params['pnid'], "{pnid}")
            
            # map the node ID variable
            if 'nodeId' in service.keys():
                if len(service['nodeId']) > 0:
                    service['nodeId'] = "{lduid}"

            # map the site ID variable
            if "siteId" in service.keys(): 
                if len(service['siteId']) > 0:
                    service['siteId'] = "{siteid}"
            
            # map the relevant unique service attribute variables.
            for attribute in service["serviceAttributes"]:
                
                if attribute['key'] == "com.vmware.cis.cm.HostId":
                    attribute['value'] = "{machineid}"

                if attribute['key'] == "com.vmware.vim.vcenter.instanceName":
                    attribute['value'] = "{pnid}"
            
            # map the relevant unique service endpoint variables.
            for endpoint in service['serviceEndpoints']:
                if self.params['os_type'] == 'windows':
                    winhome = os.environ['VMWARE_CIS_HOME']
                    windata = os.environ['VMWARE_DATA_DIR']
                    winruntime = os.environ['VMWARE_RUNTIME_DATA_DIR'].replace("\\runtime","") + "\\"
                    wincfg = os.environ['VMWARE_CFG_DIR'].replace('\\','/')
                    if "endpointAttributes" in endpoint.keys():
                        for attribute in endpoint['endpointAttributes']:

                            if winhome in attribute['value']:
                                # print(attribute['value'])
                                attribute['value'] = attribute['value'].replace(winhome, "{winhome}")
                            elif windata in attribute['value']:
                                # print(attribute['value'])
                                attribute['value'] = attribute['value'].replace(windata, "{windata}")
                            elif winruntime in attribute['value']:
                                attribute['value'] = attribute['value'].replace(winruntime, "{winruntime}")
                            elif wincfg in attribute['value']:
                                attribute['value'] = attribute['value'].replace(wincfg, "{wincfg}")
                            else:
                                continue

                if "endpointAttributes" in endpoint.keys():
                    for attribute in endpoint['endpointAttributes']:
                        if self.params['pnid'] in attribute['value']:
                            attribute['value'] = attribute['value'].replace(
                                self.params['pnid'], "{pnid}")

                if endpoint["url"]:
                    endpoint["url"] = endpoint["url"].replace(self.params['pnid'], "{pnid}")
                    
                    if self.params['domain_name'] in endpoint["url"]:
                        endpoint["url"] = endpoint["url"].replace(self.params['domain_name'], "{domain_name}")
                if endpoint["sslTrust"] and len(endpoint.get("sslTrust")) > 0:
                    endpoint['sslTrust'] = "{ssltrust}"

        # dump template to file
        if self.template_file_override:
            template_file = self.template_file_override
        else:
            template_file = self.template_file
        with open(template_file, "w") as f:
            json.dump(self.services, f, indent=1)
            logger.info("Dumped template to %s" % template_file)
        
        # return template as json
        return json.dumps(self.services, indent=1)
    
    def serviceSelect(self, rebuild_file, servicetype=""):
        """
        Provides a menu to select a service by service type
        
        Args:
            rebuild_file (str): Path to json file containing candidate
            servicetype (str, optional): The service type that has been selected
        
        Returns:
            str: the service type name and the service data for the selected service.
        """
        try:
            logger.info("Listing services...")
            rebuilddata = loadJson(rebuild_file)
        except:
            logger.info("Got list of services.")
            rebuilddata = json.loads(rebuild_file)
        count = -1
        service_list = []
        selection = servicetype
        if not selection:
            print("")
            for x in rebuilddata:
                count = count +1
                service_list.append(x['serviceType']['type'])
                print("[%s] %s " % (str(count), x['serviceType']['type']))
            print("")
            while True:
                try:
                    ans_file = input("Select number: ")
                except:
                    ans_file = raw_input("Select number: ")
                
                if int(ans_file) > count:
                    print("Invalid selection.")
                    continue
                else:
                    selection = service_list[int(ans_file)]
                    break

        service_data = None
        for x in rebuilddata:
            if selection in x['serviceType']['type']:
                service_data = x
                break
        if not service_data:
            logger.error("Failed to find servicetype '%s' in template "
                         "file data" % selection)
            raise RuntimeError("Unknown service type '%s'" % selection)

        return service_data, selection

    def rebuild_single_service(self, servicedata, servicetype):
        """
        Allows the rebuild of a single service

        Args:
            servicetype (str): service type you wish to rebuild
            servicedata (dict): service registration data
        """
        legacy_sso = ["sso:sts","sso:admin","sso:groupcheck"]
        service_id = self.getServiceId(servicetype)
        
        try:
            if "cs.identity" in servicetype:
                self.unregisterServiceType(servicetype)
                logger.info("Creating new cs.identity service...")
                ssoentries = recreateSSO(self.params['pnid'],self.cert_backup)
                if ssoentries.modern() == 1:
                    ssoentries = recreateSSO(self.params['pnid'],self.cert_backup)
                    if ssoentries.modern() == 0:
                        logger.info("Successfully recreated modern SSO endpoints.")
                    else:
                        logger.error("Failed to recreate modern SSO endpoints!")
                else:
                    logger.info("Successfully recreated modern SSO endpoints.")

            elif servicetype in legacy_sso:
                for service in legacy_sso:
                    self.unregisterServiceType(service)
                ssoentries = recreateSSO(self.params['pnid'],self.cert_backup)
                if ssoentries.legacy() == 1:
                    ssoentries = recreateSSO(self.params['pnid'],self.cert_backup)
                    if ssoentries.legacy() == 0:
                        logger.info("Successfully recreated legacy SSO endpoints.")
                    else:
                        logger.error("Failed to recreate legacy SSO endpoints!")
                else:
                    logger.info("Successfully recreated legacy SSO endpoints.")
            else:
                logger.debug("Replacing service type: %s with service data: %s" % (servicetype, servicedata))
                self.unregisterServiceType(servicetype)    
                self.ls.register(service_id, servicedata)
                msg = "Registered service: %s with service ID: %s" % (servicetype, service_id)
                logger.info(msg)
        except Exception as e:
            logger.error("FAILED TO REGISTER SERVICE! %s" % e)

    def build_from_template(self, template_file):
        """
        Build a candidate file based on template file.

        Args:
            template_file (str): File name of the template specified
        
        Returns:
            json: candidate LS services
        """
        # load the template file, replace necessary values and output a replacement file.
        self.backup()
        
        # load the template
        self.template = loadJson(template_file)

        self.params['vcserviceid'] = self.getServiceId("vcenterserver")
        if self.params['os_type'] == "windows":
            self.params['winruntime'] = os.environ['VMWARE_RUNTIME_DATA_DIR'].replace("\\runtime","") + "\\"
            self.params['windata'] = os.environ['VMWARE_DATA_DIR']
            self.params['winhome'] = os.environ['VMWARE_CIS_HOME']
            self.params['wincfg'] = os.environ['VMWARE_CFG_DIR'].replace('\\','/')
        
        # replace variables in template with my unique attributes
        for service in self.template:
            for entry in service.keys():
                
                # replace service attribute values in template
                if 'serviceAttributes' in entry:
                    for attribute in service['serviceAttributes']:
                        attribute['value'] = attribute.get('value').format(**self.params)
                    continue
                
                # replace service endpoint values in template
                if 'serviceEndpoints' in entry:
                    for endpoint in service['serviceEndpoints']:
                        if len(endpoint.get("sslTrust")) > 0:
                            endpoint['sslTrust'] = self.params['ssltrust']
                        endpoint['url'] = endpoint.get('url').format(**self.params)

                        if 'endpointAttributes' in endpoint.keys():
                            for attribute in endpoint['endpointAttributes']:
                                attribute['value'] = attribute.get('value').format(**self.params)
                    continue

                # find service IDs based on rereg directory or based on the vcserviceid
                if entry == "serviceId":
                    if 'vcserviceid' in service.get('serviceId'):
                        try:
                            serviceid = service.get('serviceId').replace("{vcserviceid}",self.params.get('vcserviceid'))
                        except:
                            print(service.get('serviceId'))
                            print(self.params)
                            sys.exit()
                    else:
                        serviceid = self.getServiceId(service['serviceType'].get('type'))
                        if len(serviceid) == 0:
                           serviceid = str(uuid.uuid4())
                           logger.debug("Creating new serviceId for serviceType: %s" % service['serviceType'])
                    service['serviceId'] = serviceid
                    continue
                
                #replace anything else found matching the variable names with params
                if "serviceType" not in entry:
                    try:
                        service[entry] = service[entry].format(**self.params)
                    except:
                        continue

        # dump newly created services to JSON file 
        with open(self.candidate_file,"w") as f:
            json.dump(self.template,f,indent=1)
            logger.debug("Dumped replacement services to %s" % self.candidate_file)
        
        # return the raw services data as JSON
        built_template = json.dumps(self.template, indent=1)
        return built_template

    def rebuild_services(self, rebuild_file):
        """
        Removes all existing service registrations for this node,
        then registers the candidate services specified by the provided
        rebuild file.  Will use the lookup service installer to regenerate
        SSO services if the node is an external PSC or embedded node.
        
        Args:
            rebuild_file (str): patht to he candidate file.
        """
        # Replaces existing services with the template data provided.
        if self.params['os_type'] == "windows":
            try:
                rebuilddata = loadJson(rebuild_file)
            except:
                rebuilddata = json.loads(rebuild_file)
        else:
            if os.path.isfile(rebuild_file):
                rebuilddata = loadJson(rebuild_file)
            else:
                rebuilddata = json.loads(rebuild_file)
        
        # unregister all services related to my pnid
        # then register the services provided by rebuilddata
        self.ls.reregister(self.params['pnid'], rebuilddata)

        # if the node is an external PSC or a VC with embedded PSC,
        # regenerate the SSO endpoints.
        psc_embedded = ['infrastructure','embedded']
        if self.params['deploytype'] in psc_embedded:           
            ssoentries = recreateSSO(self.params['pnid'],self.cert_backup)
            logger.info("Recreating SSO service registrations...")
            if ssoentries.modern() == 1:
                ssoentries = recreateSSO(self.params['pnid'],self.cert_backup)

                if ssoentries.modern() == 0:
                    logger.info("Successfully recreated modern SSO endpoints.")
                else:
                    logger.error("Failed to recreate modern SSO endpoints!")

                if ssoentries.legacy() == 0:
                    logger.info("Successfully recreated legacy SSO endpoints.")
                else:
                    logger.error("Failed to recreate legacy SSO endpoints!")

            else:
                logger.info("Successfully recreated modern SSO endpoints.")
                
                if ssoentries.legacy() == 0:
                    logger.info("Successfully recreated legacy SSO endpoints.")
                else:
                    logger.error("Failed to recreate legacy SSO endpoints!")
    
    def autoRebuild(self):
        """
        Automatically selects the correct template file.  Calls 
        the build_from_template function, followed by rebuild_services
        then registers the candidate services specified by the provided
        rebuild file.  Will use the lookup service installer to regenerate
        SSO services if the node is an external PSC or embedded node.
        """
        self.template_file = _setOtherFilename(self.templatepath, self.version, ".json", expand=False)
        warningmsg = """
            You have selected a Rebuild function.  This is a potentially destructive operation!
            All external solutions and 3rd party plugins that register with the lookup service may 
            have to be re-registered.  For example: SRM, vSphere Replication, NSX Manager, etc.
            """
        logger.debug("Detected version %s.  Attempting to choose correct template: %s " % (self.version, self.template_file))
        if os.path.isfile(self.template_file):
            logger.info("Found %s" % self.template_file)
            newdata = self.build_from_template(self.template_file)
            print(warningmsg)
            try:
                confirm = raw_input("Are you sure you want to continue?[y/n]")
            except:
                confirm = input("Are you sure you want to continue?[y/n]")
            
            if confirm.lower() != 'y':
                logger.error("Thank you for being cautious!  Please select another option.")
                self.menu()
            else:
                logger.debug("User confirmed and acknowledged risk to replace services.")
                self.rebuild_services(newdata)
        else:
            print("\nNo template found for %s.  Proceeding to file select.\n" % self.version)
            candidate_file = fileSelect(self.templatepath)
            if candidate_file:
                # newdata = candidate_file
                newdata = self.build_from_template(candidate_file)
                print(warningmsg)
                try:
                    confirm = raw_input("Are you sure you want to continue?[y/n]")
                except:
                    confirm = input("Are you sure you want to continue?[y/n]")
                
                if confirm.lower() != 'y':
                    logger.error("Thank you for being cautious!  Please select another option.")
                    self.menu()
                else:
                    logger.debug("User confirmed and acknowledged risk to replace services.")
                    self.rebuild_services(newdata)
            
            else:
                logger.warning("invalid selection!  Try again.")
            self.menu()

    def menu(self):
        """
        Provides menu for navigating the various rebuild related functions
        """
        VERSION = getVersion()
        warningmsg = """
            You have selected a Rebuild function.  This is a potentially destructive operation!
            All external solutions and 3rd party plugins that register with the lookup service may 
            have to be re-registered.  For example: SRM, vSphere Replication, NSX Manager, etc.
            """
        menu_text = '''
        Version Detected 
            Deployment type: %s
            Version: %s
        ========================

        0.  Exit 
        1.  Generate a template.
        2.  Replace all services with new services.
        3.  Replace individual service.
        4.  Restore services from backup file.

        ========================
        ''' % (self.params['deploytype'],VERSION)

        exit_code = False
        action = None
        while action not in {"1", "2", "3", "4", "0"}:
            print(menu_text)
            try:
                action = raw_input("Please select an action: ")
            except:
                action = input("Please select an action: ")
        
        if action == "1":
            self.generate_template()
            self.menu()
        
        elif action == "2":
            # candidate_file = fileSelect(self.candidatepath)
            self.autoRebuild()
            self.menu()

        elif action == "3":
            # logger.info("\nPLEASE NOTE:  This does not support recreating the SSO endpoints.  You must perform a full rebuild for that.\n")
            # candidate_file = fileSelect(self.candidatepath)
            candidate_file = self.getTemplate()
            if candidate_file:
                print(warningmsg)
                try:
                    servicedata, servicetype = self.serviceSelect(candidate_file)
                except:
                    logger.exception("Couldn't get service data!")
                    self.menu()
                msg = "You have chosen to replace the service: %s -- Continue?[y/n]: " % servicetype
                try:
                    confirm = raw_input(msg)
                except:
                    confirm = input(msg)
                
                if confirm.lower() != 'y':
                    logger.error("Thank you for being cautious!  Please select another option.")
                    self.menu()
                else:
                    logger.debug("User confirmed and acknowledged risk to replace the %s service." % servicetype)
                    self.rebuild_single_service(servicedata, servicetype)
            self.menu()

        elif action == "4":

            backup_file = fileSelect(self.backuppath)
            if backup_file:
                print(warningmsg)
                try:
                    confirm = raw_input("Are you sure you want to continue?[y/n]")
                except:
                    confirm = input("Are you sure you want to continue?[y/n]")
                
                if confirm.lower() != 'y':
                    logger.error("Thank you for being cautious!  Please select another option.")
                    self.menu()
                else:
                    logger.debug("User confirmed and acknowledged risk to replace services.")
                    self.rebuild_services(backup_file)
            
            else:
                logger.warning("No files in backup directory.  Please generate a template first.")
                self.menu()

        else:
            logger.info("Exiting...")

    def run_action(self):
        if not self.rebuild_action:
            logger.error("A rebuild action was not specified for the rebuilder")
            raise RuntimeError("No rebuild action specified")

        if self.rebuild_action == ACTION_CREATE_TEMPLATE:
            logger.debug("Running rebuild action to generate a new "
                         "template file")
            self.generate_template()

        elif self.rebuild_action == ACTION_REBUILD_SERVICE:
            if not self.service_override:
                logger.error("A service name was not specified for the service "
                             "rebuild action")
                raise ValueError("Missing service name for rebuild")

            logger.debug("Running rebuild action to rebuild specified "
                         "service %s" % self.service_override)
            template_data = self.getTemplate(prompt=False)
            servicedata, servicetype = self.serviceSelect(template_data,
                                                          self.service_override)
            logger.debug("Rebuilding and replacing service %s" % servicetype)
            self.rebuild_single_service(servicedata, servicetype)

        elif self.rebuild_action == ACTION_REBUILD_ALL_SERVICES:
            logger.debug("Running rebuild action to rebuild all services")
            template_data = self.getTemplate(prompt=False)
            self.rebuild_services(template_data)

        elif self.rebuild_action == ACTION_RESTORE:
            if not self.backup_file_override:
                logger.error("A backup file was not specified for the "
                             "restore action")
                raise ValueError("Missing backup file path for restore action")
            if not os.path.isfile(self.backup_file_override):
                logger.error("The specified backup path is not a file "
                             "on the system: %s" % self.backup_file_override)
                raise RuntimeError("Backup file path does not exist: %s"
                                   % self.backup_file_override)

            logger.debug("Running rebuild action to restore from backup "
                         "file %s" % self.backup_file_override)
            self.rebuild_services(self.backup_file_override)

        else:
            logger.error("The following rebuild action is not recognized: %s"
                           % self.rebuild_action)
            raise RuntimeError("Unrecognized rebuild action: %s"
                               % self.rebuild_action)











