#!/usr/bin/env python
"""
You must run this on any PSC in the same SSO site as the LB, as all PSC's in a single SSO site should be 
behind the load balancer.

Assumptions:

Customer has configured (or attempted to configure) PSC HA using the documentation we have available.
This portion attempts to reverse the same steps in those scripts.

This option performs the following tasks:

    -- Updates the hostname.txt file with the PSC's FQDN (replaces the LB VIP FQDN)
        -- During PSC HA configuration, this file is updated to reflect LB VIP FQDN
        -- Outside of PSC HA, this file should always contain the PNID/PSC FQDN
    
    -- Removes 'proxyName' element from the STS server.xml file
        -- During PSC HA configuration, this file is updated to reflect LB VIP FQDN
        -- This element doesn't exist without PSC HA being configured.
    
    -- Replaces vmdir cert and key if it exists (with MACHINE cert)
        -- in 6.0, the vmdir certificate is replaced during PSC HA configuration.
        -- Replaces this certificate with the same from MACHINE_SSL_CERT
    
    -- Re-registers the cs.license service registration with the FQDN and SSL cert of the PSC
        -- Utilizes lookup service API and transforms to put into dictionary, updates the values, transforms 
           back to api spec and re-registers
        -- Uses PNID for FQDN and MACHINE_SSL_CERT (returned by params function in utils.py)
    
    -- Removes all the SSO service registrations referencing the LB VIP FQDN
        -- Searches all lookup service registrations for ones referencing LB VIP (found in hostname.txt)
        -- Unregisters them
    
    -- Recreates the SSO service registrations with current SSL certificate and FQDN of the PSC using the 
       lookup service installer
        -- The paths to java libs change between versions and platforms.  Tries the various configs until one works
"""
import os
import platform
from .utils import *
from .stale import VecsCheck
import xml.etree.ElementTree as ET
import shutil
try:
    import httplib
except ImportError:
    import http.client as httplib
try:
    import urllib.parse as urlparse
    from urllib.request import Request, urlopen
    from urllib.error import URLError, HTTPError
except ImportError:
    import urlparse
    from urllib2 import Request, urlopen
    from urllib2 import URLError, HTTPError

curpath = os.getcwd()

import logging

logger = logging.getLogger(__name__)

if not os.environ['VMWARE_PYTHON_PATH'] in sys.path:
    sys.path.append(os.environ['VMWARE_PYTHON_PATH'])
from cis.utils import read_properties, replace_properties_in_file

class changeLocal(object):
    '''
    This class reverses the changes made by updateSSOconfig.py, which changes local 
    parameters on the node (not in the lookup service)
    
    Attributes:
        hostname_file (str): path to hostname.txt
        pnid (str): primary network identifier of current node (FQDN)
        server_xml_file (str): STS server.xml file path
        service_control_exe (str): path service-control 
        vmdir_cert_file (str): vmdir cert file path
        vmdir_key_file (str): vmdir key file path
    
    '''
    def __init__(self, pnid):
        """
        Args:
            pnid (TYPE): Description
        """
        self.pnid = pnid
        self.proxyname_propkey = "vmidentity.server.connector.ssl.proxy-name"

        if os.name == 'posix':
            self.hostname_file = "/etc/vmware-sso/hostname.txt"
            self.server_xml_file = "/usr/lib/vmware-sso/vmware-sts/conf/server.xml"
            self.catalina_prop_file = "/usr/lib/vmware-sso/vmware-sts/conf/catalina.properties"
            self.service_control_exe = "/bin/service-control"
            self.vmdir_cert_file = "/usr/lib/vmware-vmdir/share/config/vmdircert.pem"
            self.vmdir_key_file = "/usr/lib/vmware-vmdir/share/config/vmdirkey.pem"

        else:
            self.hostname_file = os.environ['VMWARE_CFG_DIR'] + "\\sso\\hostname.txt"
            self.server_xml_file = os.environ['VMWARE_RUNTIME_DATA_DIR'] + "\\VMwareSTSService\\conf\\server.xml"
            self.service_control_exe = os.environ['VMWARE_CIS_HOME'] + "\\bin\\service-control.bat"
            self.vmdir_cert_file = os.environ['VMWARE_CFG_DIR'] + "\\vmdird\\vmdircert.pem"
            self.vmdir_key_file = os.environ['VMWARE_CFG_DIR'] + "\\vmdird\\vmdirkey.pem"

    def mod_hostnamefile(self):
        """
        hostname.txt file doesn't exist in 6.7, so we skip it if we can't find it.
        """
        if os.path.exists(self.hostname_file):
            logger.info("Modifying hostname.txt")
            try:
                with open(self.hostname_file,"w") as fd:
                    fd.write(self.pnid)
                    logger.debug("Wrote %s to hostname file." % self.pnid)
            except:
                logger.warning("Couldn't write to hostname file!")
                pass
        else:
            logger.info("no hostname.txt file.")

    def unconfigure_proxyName(self):
        """
        Unconfigure 'proxyName' from STS connector
        """
        properties = read_properties(self.catalina_prop_file, [self.proxyname_propkey])

        # For 9.0, the configuration of server.xml is externalized to catalina.properties.
        # the vmidentity.server.connector.ssl.proxy-name properties only exist in catalina.properties from 9.0
        if os.name == 'posix' and self.proxyname_propkey in properties:
            logger.info("Currently, proxy name is configured as '%s'." % properties.get(self.proxyname_propkey))
            replace_properties_in_file(self.catalina_prop_file, {self.proxyname_propkey: ""})
            logger.info("Unconfigure value of proxy name")
        else:
            self.mod_serverxmlfile()

    def mod_serverxmlfile(self):
        """
        Removes 'proxyName' element from STS server.xml
        """
        logger.info("Checking server.xml file for LB VIP")
        tree = ET.parse(self.server_xml_file)
        root = tree.getroot()
        flag = False
        for entry in root:
            if entry.tag == 'Service' and entry.attrib['name'] == 'Catalina' :
                for child in entry :
                    if child.tag == 'Connector' and child.get("SSLEnabled") != None:
                        if child.get('proxyName'):
                            flag = True
                            logger.info("proxyName attribute found.  Removing LB proxy attribute.")
                            del child.attrib['proxyName']
                        break
        if flag == True:
            tree.write(self.server_xml_file)
        else:
            logger.info("No LB VIP configured in server.xml file.")

    def mod_vmdircert(self, exportpath):
        """
        replaces vmdir cert/key with MACHINE_SSL_CERT cert/key
        
        Args:
            exportpath (str): Path to export directory
        """
        cert = VecsCheck(exportpath)
        machinecert, machinekey = cert.backup_machine()
        if os.path.exists(self.vmdir_cert_file):
            logger.info("vmdir cert found.  Replacing...")
            shutil.copyfile(machinecert,self.vmdir_cert_file)
            shutil.copyfile(machinekey,self.vmdir_key_file)
            logger.info("Successfully replaced vmdir cert.")
        else:
            logger.info("%s does not exist.  Skipping." % self.vmdir_cert_file)
    
    def execute(self, exportpath):
        """
        Executes the check
        
        Args:
            exportpath (str): Path to export directory
        """
        self.mod_hostnamefile()
        self.unconfigure_proxyName()
        self.mod_vmdircert(exportpath)

class PscHa(object):
    """
    Reverses the configuration for PSC HA.
    
    Attributes:
        cert_file (TYPE): Description
        change_local (TYPE): Description
        exportpath (TYPE): Description
        ls (TYPE): Description
        node_services (TYPE): Description
        pnid (TYPE): Description
        site_services (TYPE): Description
        ssltrust (TYPE): Description
        sso_hostname (TYPE): Description
    """
    #  Gets all service registrations for this SSO site and node
    def __init__(self, params, username, password, exportpath=curpath):
        """
        Args:
            params (dict): dictionary of node specific parameters
            username (str): SSO admin username
            password (str): SSO admin password
            exportpath (str, optional): Path to export directory
        """
        # Initiate lookup service client
        self.exportpath = exportpath

        self.pnid = params['pnid']
        self.ssltrust = params['ssltrust']
        cert = VecsCheck(exportpath)
        cert.backup_machine()

        self.cert_file = cert.machine_cert_file
        self.change_local = changeLocal(self.pnid)
        self.sso_hostname =  params['sso_hostname']

        try:
            self.ls = LookupServiceClientHelper(params['psc'], username=username, password=password)
            logger.info("Established LS connection to %s" % params['psc'])
        except (sso.SoapException,Exception) as e:
            if 'Invalid credentials' in str(e):
                sys.exit()

            logger.error("Initial connection to lookup service failed.  Will attempt to regenerate cs.identity and try again")
            try:
                cert = VecsEntry("MACHINE_SSL_CERT")
                exported = cert.get_cert("__MACHINE_CERT",self.cert_file)
                psc_embedded = ['infrastructure','embedded']
                if params['deploytype'] in psc_embedded:           
                    ssoentries = recreateSSO(params['pnid'],self.cert_file)
                    logger.warning("Recreating SSO service registrations...")
                    if ssoentries.modern() == 1:
                        ssoentries = recreateSSO(params['pnid'],self.cert_file)
                        if ssoentries.modern() == 0:
                            logger.info("Successfully recreated modern SSO endpoints.")
                        else:
                            logger.exception("Failed to recreate modern SSO endpoints!")
                            raise
                    self.ls = LookupServiceClientHelper(params['psc'], username=username, password=password)
                    logger.info("Established LS connection to %s" % params['psc'])
            except Exception as e:
                logger.exception("Still failed to connect to lookup service.  Further investigation required.  error: %s" % e)
                raise
        
        # getServices only returns services for this node.
        self.node_services = self.ls.getNode(params['lduid'])
        logger.info("Retrieved services for machine with node ID: %s" % params['lduid'])
        self.site_services = self.ls.getSite(params['siteid'])
        logger.info("Retrieved services from SSO site: %s" % params['siteid'])

    def replaceLicense(self):
        """
        This updates the SSL cert and hostname for cs.license
        We can do this because it has a Node ID.
        """ 
        logger.info("Reregistering cs.license")
        foundflag = False

        for service in self.node_services:
            if service['serviceType']['type'] == 'cs.license':
                foundflag = True
                serviceId = service.get('serviceId')
                logger.info("cs.license service has service ID: %s" % serviceId)
                for endpoint in service['serviceEndpoints']:
                    endpoint['sslTrust'] = self.ssltrust
                    logger.debug("Found %s" % endpoint['url'])
                    oldurl = urlparse.urlparse(endpoint['url'])

                    newhostname = self.pnid + ':' + str(oldurl.port)
                    newurl = oldurl._replace(netloc=newhostname)
                    newurl = urlparse.urlunparse(newurl)
                    logger.debug("Replacing with %s" % newurl)
                    endpoint['url'] = newurl


                logger.debug("Unregistering %s" % serviceId)
                self.ls.unregister(serviceId)
                logger.debug("Registering %s" % serviceId)
                self.ls.register(serviceId,service)

        if foundflag == False:
            logger.error("No cs.license service detected!  You will have to rebuild it from scratch.")
        logger.info("Finished.")
    
    def checkSsoServices(self):
        """
        Since the SSO services don't have a node ID, we will remove and recreate
        all of them using LS installer.
        """
        myservices, hostid = self.ls.getPnid(self.pnid)
        ssoservices = ['cs.identity','sso:sts','sso:admin','sso:groupcheck']
        sso_service_list = []
        for service in myservices:
            serviceType = service['serviceType']['type']
            serviceId = service['serviceId']
            if serviceType in ssoservices:
                logger.debug("Found %s with service ID: %s" % (serviceType,serviceId))
                sso_service_list.append(serviceId)
        
        if len(sso_service_list) > 0:
            logger.info("Unregistering SSO services...")
            for serviceid in sso_service_list:
                self.ls.unregister(serviceid)
                logger.debug("Unregistered %s" % serviceid)
        
        logger.info("Recreating SSO service registrations...")
        regen = recreateSSO(self.pnid,self.cert_file)
        if regen.modern() == 1:
            regen = recreateSSO(self.pnid,self.cert_file)
            if regen.modern() == 0:
                logger.info("Successfully recreated modern SSO endpoints.")
            if regen.legacy() == 0:
                logger.info("Successfully recreated legacy SSO endpoints.")

        else:
            logger.info("Successfully recreated modern SSO endpoints.")
            if regen.legacy() == 0:
                logger.info("Successfully recreated legacy SSO endpoints.")

    def removeLb(self):
        """
        We detect the LB from hostname.txt file and unregister services
        with the LB VIP FQDN.  We skip cs.license as we want to ensure
        we don't remove ones that haven't been re-registered yet.
        """
        if self.sso_hostname != self.pnid:
            logger.info("Load Balancer Detected!  Attempting to unregister the LB services")
            lbservices, hostid = self.ls.getPnid(self.sso_hostname)
            if len(lbservices) > 0:
                logger.info("LB VIP services found!")
                for service in lbservices:
                    if 'cs.license' not in service['serviceType']['type']:
                        serviceId = service['serviceId']
                        logger.debug("Unregistering %s with serviceId: %s" % (service['serviceType']['type'],serviceId))
                        self.ls.unregister(serviceId)
            else:
                logger.info("No LB VIP services found in the lookup service.")
        else:
            logger.info("This machine is not locally configured for a LB VIP.")
        logger.info("Finished.")

    def check(self):
        """
        Execute the check
        """
        self.replaceLicense()
        self.removeLb()
        self.checkSsoServices()
        self.change_local.execute(self.exportpath)



    

