#!/usr/bin/env python
"""
Calls lstool_parse and lstool_scan to spit out a report of any common
problems found in the lookup service across the entire SSO domain.  Will
output list of problems found.  Does not make changes.
"""

from .utils import *
from .lstool_scan import *
from .lstool_parse import *
from .utils import _getSslCert
import json
import logging

logger = logging.getLogger(__name__)

class lsReport(object):

    """
    Runs lstool_parse and lstool_scan to identify problems in the SSO domain.
    
    Attributes:
        report (json): this is the output of lstool_scan.py
        report_file (str): path to destination scan results file.
    """

    def __init__(self, params, report_file):
        """
        Args:
            params (dict): dictionary of local node parameters returned from 
            utils.get_params()
            report_file (str): path to destination scan results file.
        """
        parser = LSTool_Parse()
        lsJsonData = parser.parseData()
        scanner = LSTool_Scan(lsJsonData)
        self.report = scanner.execute(live=True)
        self.report_file = report_file

    def generateReport(self):
        """
        This function outputs the problems found (if any) and dumps the report to self.report_file
        """
        logger.info("Listing lookup service problems found in SSO domain")
        for site in self.report:
            for node in self.report[site]:
                result = self.report[site][node]['Problems Detected']
                if result != "No problems found.":
                    for problem in self.report[site][node]['Problems Detected']:
                        if problem:
                            issues = problem + ": " + self.report[site][node]['Problems Detected'][problem].get('Recommended Action')
                            logger.error("%s\\%s found %s " % (site, node, issues))
                else:
                    logger.info("No issues detected in the lookup service entries for %s." % node)
        with open(self.report_file,'w+') as out_json:
            try:
                json.dump(self.report, out_json, sort_keys=True, indent=4)
            except TypeError as e:
                logger.error("There was a problem exporting the report to a file.  Please provide the contents of %s to VMware support.  Dumping raw data to the report instead.")
                logger.error("Error was: %s" % e)
                out_json.write(self.report)
        logger.info("Report generated:  %s" % self.report_file)


